package com.ai.ble.base.util;

import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.*;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ObjectNode;


public final class JsonUtil {

	private static ObjectMapper objectMapper = new ObjectMapper();
	private static JsonFactory jsonFactory = new JsonFactory();

	static {
		objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
		//objectMapper.configure(SerializationFeature.WRITE_NULL_MAP_VALUES, false);
		objectMapper.setDateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
		objectMapper.setSerializationInclusion(Include.NON_NULL);
		objectMapper.configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, true);
	}

	public static <T> T fromJson(String jsonAsString, Class<T> pojoClass) throws JsonMappingException,
			JsonParseException, IOException {
		return objectMapper.readValue(jsonAsString, pojoClass);
	}
	public static <T> List<T> fromJsonToList(String jsonAsString, Class<T> pojoClass)
			throws JsonParseException, JsonMappingException, IOException{
		return objectMapper.readValue(
				jsonAsString,
				objectMapper.getTypeFactory().constructCollectionType(List.class, pojoClass));
	}

	public static <T> T fromJson(FileReader fr, Class<T> pojoClass) throws JsonParseException, IOException {
		return objectMapper.readValue(fr, pojoClass);
	}

	public static String toJson(Object pojo) throws JsonMappingException, JsonGenerationException, IOException {
		return toJson(pojo, false);
	}

	public static String toJson(Object pojo, boolean prettyPrint) throws JsonMappingException, JsonGenerationException,
			IOException {
		StringWriter sw = new StringWriter();
		JsonGenerator jg = jsonFactory.createGenerator(sw);
		if (prettyPrint) {
			jg.useDefaultPrettyPrinter();
		}
		objectMapper.writeValue(jg, pojo);
		return sw.toString();
	}

	public static void toJson(Object pojo, FileWriter fw, boolean prettyPrint) throws JsonMappingException,
			JsonGenerationException, IOException {
		JsonGenerator jg = jsonFactory.createGenerator(fw);
		if (prettyPrint) {
			jg.useDefaultPrettyPrinter();
		}
		objectMapper.writeValue(jg, pojo);
	}

	@SuppressWarnings("unchecked")
	public static Map<String, Object> parseMap(String jsonStr) throws IOException {
		Map<String, Object> map = objectMapper.readValue(jsonStr, Map.class);
		return map;
	}

	public static JsonNode parse(String jsonStr) throws IOException {
		JsonNode node = null;
		node = objectMapper.readTree(jsonStr);
		return node;
	}

	public static ObjectMapper getObjectMapper() {
		return new ObjectMapper();
	}
	public static ObjectNode getObjectNode() {
		return getObjectMapper().createObjectNode();
	}
	public static String convertNode( ObjectNode node) throws JsonProcessingException {
		final Object obj = objectMapper.treeToValue(node, Object.class);
		final String json = objectMapper.writeValueAsString(obj);
		return json;
	}
}
