package com.ai.ble.base.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * 时间格式工具类
 *
 * @author  CaoHang
 * @version  [版本号, 2014-11-25]
 * @see  [相关类/方法]
 * @since  [产品/模块版本]
 */
public final class TimestampUtil
{
	private static Logger logger = LogManager.getLogger(LogManager.ROOT_LOGGER_NAME);
	private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT_DEFAULT = new ThreadLocal<SimpleDateFormat>(){
		@Override
		protected SimpleDateFormat initialValue(){
			return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		}
	};
	private static final ThreadLocal<SimpleDateFormat> DATEFORMAT_YYYYMMDDHHMMSSSSS = new ThreadLocal<SimpleDateFormat>(){
		@Override
		protected SimpleDateFormat initialValue(){
			return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss SSS");
		}
	};
	private static final ThreadLocal<SimpleDateFormat> DATEFORMAT_YYYYMMDDHHMMSS = new ThreadLocal<SimpleDateFormat>(){
		@Override
		protected SimpleDateFormat initialValue(){
			return new SimpleDateFormat("yyyyMMddHHmmss");
		}
	};
	private static final ThreadLocal<SimpleDateFormat> DATEFORMAT_YYYYMMDD = new ThreadLocal<SimpleDateFormat>(){
		@Override
		protected SimpleDateFormat initialValue(){
			return new SimpleDateFormat("yyyy-MM-dd");
		}
	};

	private static final ThreadLocal<SimpleDateFormat> DATEFORMAT_YYYYMM = new ThreadLocal<SimpleDateFormat>(){
		@Override
		protected SimpleDateFormat initialValue(){
			return new SimpleDateFormat("yyyy-MM");
		}
	};
	private static final ThreadLocal<SimpleDateFormat> DATE_FORMAT_DEFAULT_WITH_T = new ThreadLocal<SimpleDateFormat>(){
		@Override
		protected SimpleDateFormat initialValue(){
			return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
		}
	};

    /**
     * 获取指定格式的时间格式化对象
     *
     * @param format 时间格式
     * @return DateFormat 时间格式化对象
     * @see [类、类#方法、类#成员]
     */
    public static DateFormat getDateFormater(String format)
    {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf;
    }

    /**
     * 获取固定格式的时间格式化对象
     * <li>格式：yyyy-MM-dd HH:mm:ss</li>
     *
     * @return DateFormat 时间格式化对象
     * @see [类、类#方法、类#成员]
     */
    public static DateFormat getDateFormater()
    {
        return DATE_FORMAT_DEFAULT.get();
    }

    /**
     * 获取固定格式的时间戳
     * <li>格式：yyyy-MM-dd HH:mm:ss</li>
     *
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String getTimeStamp(String format)
    {
        return getDateFormater(format).format(new Date());
    }

    /**
     * 获取固定格式的时间戳
     * <li>格式：yyyy-MM-dd HH:mm:ss</li>
     *
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String getTimeStamp()
    {
        return DATE_FORMAT_DEFAULT.get().format(new Date());
    }
    /**
     * 获取固定格式的时间戳
     * <li>格式：yyyy-MM-dd HH:mm:ss sss</li>
     *
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String getTimeStampYYYYMMDDHHMMSSSSS()
    {
        return DATEFORMAT_YYYYMMDDHHMMSSSSS.get().format(new Date());
    }
    /**
     * 获取固定格式的时间戳
     * <li>格式：yyyyMMddHHmmss</li>
     *
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String getTimeStampYYYYMMDDHHMMSS()
    {
        return DATEFORMAT_YYYYMMDDHHMMSS.get().format(new Date());
    }
    /**
     * 获取固定格式的时间戳
     * <li>格式：yyyy-MM-dd</li>
     *
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String getTimeStampByYYYYMMDD()
    {
        return DATEFORMAT_YYYYMMDD.get().format(new Date());
    }

    /**
     * 获取固定格式的时间戳
     * <li>格式：yyyy-MM</li>
     *
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String getTimeStampByYYYYMM()
    {
        return DATEFORMAT_YYYYMM.get().format(new Date());
    }

    /**
     * 解析指定格式的时间戳为时间对象
     * <li>格式：yyyy-MM-dd HH:mm:ss</li>
     *
     * @param timeStamp 时间戳
     * @return Date 时间对象
     * @see [类、类#方法、类#成员]
     */
    public static Date parserToDate(String timeStamp)
        throws ParseException
    {
        return DATE_FORMAT_DEFAULT.get().parse(timeStamp);
    }

    /**
     * 解析指定格式的时间戳为时间对象
     * <li>格式：yyyy-MM-dd</li>
     *
     * @param timeStamp 时间戳
     * @return Date 时间对象
     * @see [类、类#方法、类#成员]
     */
    public static Date parserToDateToYYYYMMDD(String timeStamp)
        throws ParseException
    {
        return DATEFORMAT_YYYYMMDD.get().parse(timeStamp);
    }

    /**
     * 解析时间对象为固定格式的时间戳
     * <li>格式：yyyy-MM-dd HH:mm:ss</li>
     *
     * @param date 日期对象
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String parserDateToString(Date date)
        throws ParseException
    {
        return DATE_FORMAT_DEFAULT.get().format(date);
    }

    /**
     * 获取默认的日期对象
     *
     * @return 日期对象
     * @see [类、类#方法、类#成员]
     */
    public static Date getDefaultDate()
    {
        return new Date();
    }

    /**
     * 获取昨日日期
     *
     * @return 日期对象
     * @see [类、类#方法、类#成员]
     */
    public static String getYesterday()
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, -1);//得到前一天
        Date date = calendar.getTime();
        String yesterday = DATEFORMAT_YYYYMMDD.get().format(date);

        return yesterday;
    }


    /**
     * 获得距离现在的某个时间点
     * @param field
     * @param mount
     * @return
     */
    public static Date getDateFromNow(int field, int mount){
    	Calendar calendar = Calendar.getInstance();
    	calendar.add(field, mount);
    	return calendar.getTime();
    }

    /**
     * 获取距今多少天的日期
     * @param days
     * 		距离今天的天数
     * @return
     */
    public static String getPastDate(int days){

    	Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, -days);//得到前一天
        Date date = calendar.getTime();
        String yesterday = DATEFORMAT_YYYYMMDD.get().format(date);

        return yesterday;
    }


    /**
     * 解析时间对象为固定格式的时间戳
     * <li>格式：yyyy-MM-dd</li>
     *
     * @param date 日期对象
     * @return String 时间戳
     * @see [类、类#方法、类#成员]
     */
    public static String parserDate(Date date)
        throws ParseException
    {
        return DATEFORMAT_YYYYMMDD.get().format(date);
    }


    /**指定字符串格式转化为指定字符串格式
     *
     */
    public static String yyyyMMdd2yyyy_MM_dd(String str)
    {

    	   String form = "";
    	   if(str.length()==8){
    		   form = "yyyyMMdd";

    	   }else if(str.length()==14){
    		   form = "yyyyMMddHHmmss";
    	   }
           //字符串转date类型
    	   SimpleDateFormat format = new SimpleDateFormat(form);
    	   Date date = null;
    	   try {
    	    date = format.parse(str);
    	   } catch (ParseException e) {
//    		   log.error(e.getMessage(),e);
    	   }
    	   SimpleDateFormat format2 = new SimpleDateFormat("yyyy-MM-dd");
    	   String str2 = format2.format(date);
    	   System.out.println(str2);
    	   return str2;
    }

    /**
     * 获取格式化Date数据
     */
    public static Date getDefaultDateByTimeStamp() {
    	try {
			return parserToDate(getTimeStamp());
		} catch (ParseException e) {
			return new Date();
		}
	}

    /**比较时间
	 * 接收yyyy-mm-dd 类型格式
	 * @param StartDate
	 * @param EndDate
	 * @return
	 */
	public static int compareDate(Date StartDate, Date EndDate) {
		try {

			if (StartDate.after(EndDate)) {
				logger.info("StartDate 在EndDate 日期后");
				return 1;
			} else if (StartDate.before(EndDate) ) {
				logger.info("StartDate 在EndDate日期前");
				return -1;
			} else {
				return 0;
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
		return 0;
	}

	/**
	 * zhangdd 两个时间之间相差距离多少天
	 *
	 * @param start 时间参数 1：
	 * @param end 时间参数 2：
	 * @return 相差天数
	 */
	public static long getDistanceDays(String start, String end) {
		DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
		Date one;
		Date two;
		long days = 0;
		try {
			one = df.parse(start);
			two = df.parse(end);
			long time1 = one.getTime();
			long time2 = two.getTime();
			long diff;
			if (time1 < time2) {
				diff = time2 - time1;
			} else {
				diff = time1 - time2;
			}
			days = diff / (1000 * 60 * 60 * 24);
		} catch (ParseException e) {
			e.printStackTrace();
		}
		return days;
	}

	/**
	 * 获取 n 天之后的时间
	 * @param startTime 开始时间
	 * @param n
	 * @return 是否过期
	 */
	public static Date getExpireDate(String startTime, int n) throws ParseException {
		Date startDate = parserToDate(startTime);
		return getExpireDate(startDate, n);
	}

	/**
	 * 获取 n 天之后的时间
	 * @param startDate 开始日期
	 * @param n
	 * @return 是否过期
	 */
	public static Date getExpireDate(Date startDate, int n) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(startDate);
		cal.add(Calendar.DATE, n);
		return cal.getTime();
	}


	public static String getTimeStampWithT(Date date)
	{
		return DATE_FORMAT_DEFAULT_WITH_T.get().format(date);
	}


	/**使用jdk1.8 中线程安全获取 yyyyMMddHHmmssSSS  格式时间
	 *
	 * @return
	 */
	public static String getYYYYMMDDHHmmssSSS() {
		LocalDateTime lt = LocalDateTime.now();
        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");
        return lt.format(df);
	}
}
